package newstate;

import java.util.*;

/**
* This is the context of a FSM. A context holds a reference to the current state
* and also functions as a repository for objects. To do that it extends from
* java.util.Hashtable. Objects can be stored with the hashtable's put method and
* retrieved with the hashtables get method.
*/
public class FSMContext extends Hashtable
{
        private State state;
        private State firstState;
        private FSMAction initialAction = State.skip;
                
        public FSMContext()
        {
        }
        
        /**
        * Create a new context with s as the first state
        * @param s The first state
        */
        public FSMContext(State s)
        {
                super();
                setFirstState(s);
                initialize();
        }
        
        /**
        * Create a new context with s as the first state. an then execute i to
        * initialize the context.
        * @param s The first state
        * @param i The initial action
        */
        public FSMContext(State s, FSMAction i)
        {
                super();
                setFirstState(s);
                setInitialAction(i);
                initialize();
        }
        
        /**
        * Find out what events can be sent to the current state
        * @return A list of events
        */
        public Vector getEvents()
        {
                return getState().getEvents();
        }
               
        
        /**
        * Initialize the context. Set first state and execute initial action.
        */
        public void initialize()
        {
                this.clear();
                initialAction.execute(this, null);
                setState(firstState);
                firstState.getStateEntryAction().execute(this, null);
        }
        
        public void setState(State s) { state = s; }
        public State getState() { return state; }
        
        public void setFirstState(State s) { firstState = s; }
        public State getFirstState() { return firstState; }
        
        public void setInitialAction(FSMAction a) { initialAction = a; }
        public FSMAction getInitialAction() { return initialAction; }
        
        /**
        * Dispatch an event e.
        * @param e The event
        - @param data Some additional data
        */
        public void dispatch(FSMEvent e, Object data)
        {
                getState().dispatch(e, data, this);
        }
}